/*
 * Decompiled with CFR 0.152.
 */
package org.smslib.queues;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.PriorityBlockingQueue;
import org.smslib.OutboundMessage;
import org.smslib.Service;
import org.smslib.helper.Logger;
import org.smslib.queues.AbstractQueueManager;
import org.smslib.queues.ScheduledOutboundMessage;

public class DefaultQueueManager
extends AbstractQueueManager {
    public static final String MESSAGE_FILE_EXT = ".msg";
    private Map<String, PriorityBlockingQueue<OutboundMessage>> queueMap;
    private DelayQueue<ScheduledOutboundMessage> delayQueue;
    private String queueDirectory;
    private File pendingMessageDir;
    private File delayedMessageDir;

    public DefaultQueueManager() {
    }

    public DefaultQueueManager(String queueDirectory) {
        this.queueDirectory = queueDirectory;
    }

    public DefaultQueueManager(int delay) {
        super(delay);
    }

    public DefaultQueueManager(int delay, String queueDirectory) {
        super(delay);
        this.queueDirectory = queueDirectory;
    }

    @Override
    protected void init() {
        super.init();
        this.queueMap = new HashMap<String, PriorityBlockingQueue<OutboundMessage>>();
        this.delayQueue = new DelayQueue();
        if (this.queueDirectory == null) {
            this.queueDirectory = Service.getInstance().getSettings().QUEUE_DIRECTORY;
            if (this.queueDirectory == null) {
                Logger.getInstance().logInfo("Queue directory not defined. Queued messages will not be saved to filesystem.", null, null);
                return;
            }
        }
        File baseDir = new File(this.queueDirectory, "queue");
        this.pendingMessageDir = new File(baseDir, "pending");
        if (!this.pendingMessageDir.exists()) {
            if (!this.pendingMessageDir.mkdirs()) {
                Logger.getInstance().logError("Could not create directory for pending messages queue at " + this.pendingMessageDir.getPath(), null, null);
            }
        } else {
            Logger.getInstance().logDebug("loading pending messages..", null, null);
            this.loadPendingMessages();
        }
        this.delayedMessageDir = new File(baseDir, "delayed");
        if (!this.delayedMessageDir.exists()) {
            if (!this.delayedMessageDir.mkdirs()) {
                Logger.getInstance().logError("Could not create directory for delayed messages queue at " + this.delayedMessageDir.getPath(), null, null);
            }
        } else {
            Logger.getInstance().logDebug("loading delayed messages..", null, null);
            this.loadDelayedMessages();
        }
    }

    @Override
    public boolean queueMessage(OutboundMessage message) {
        if (message.getDeliveryDelay() > 0L) {
            return this.addToDelayedQueue(message, true);
        }
        return this.addToGatewayQueue(message, true);
    }

    @Override
    public boolean removePendingMessage(OutboundMessage message) {
        for (PriorityBlockingQueue<OutboundMessage> q : this.queueMap.values()) {
            if (!q.remove(message)) continue;
            this.deletePendingMessage(message.getGatewayId(), message.getUuid());
            return true;
        }
        return false;
    }

    @Override
    public boolean removePendingMessage(String messageUUID) {
        for (PriorityBlockingQueue<OutboundMessage> q : this.queueMap.values()) {
            for (OutboundMessage m : q) {
                if (!m.getId().equalsIgnoreCase(messageUUID) || !q.remove(m)) continue;
                this.deletePendingMessage(m.getGatewayId(), messageUUID);
                return true;
            }
        }
        return false;
    }

    private boolean addToGatewayQueue(OutboundMessage message, boolean store) {
        PriorityBlockingQueue<OutboundMessage> queue = this.queueMap.get(message.getGatewayId());
        if (queue == null) {
            queue = new PriorityBlockingQueue<OutboundMessage>(50, new PriorityComparator());
            this.queueMap.put(message.getGatewayId(), queue);
        }
        boolean queued = queue.add(message);
        if (store && queued) {
            this.storePendingMessage(message);
        }
        return queued;
    }

    private boolean addToDelayedQueue(OutboundMessage message, boolean store) {
        boolean queued = this.delayQueue.add(new ScheduledOutboundMessage(message));
        if (store && queued) {
            this.storeDelayedMessage(message);
        }
        return queued;
    }

    private boolean storePendingMessage(OutboundMessage message) {
        if (this.queueDirectory == null) {
            return true;
        }
        File gatewayDir = new File(this.pendingMessageDir, message.getGatewayId().replace("/", "."));
        if (!gatewayDir.exists() && !gatewayDir.mkdir()) {
            Logger.getInstance().logError("Queue directory could be created for gateway " + message.getGatewayId() + ". Could not create directory .." + gatewayDir.getPath(), null, null);
            return false;
        }
        return this.serializeMessage(message, new File(gatewayDir, message.getUuid() + MESSAGE_FILE_EXT));
    }

    private boolean deletePendingMessage(String gatewayId, String messageUUID) {
        if (this.queueDirectory == null) {
            return true;
        }
        return new File(new File(this.pendingMessageDir, gatewayId), messageUUID + MESSAGE_FILE_EXT).delete();
    }

    private boolean deletePendingMessages(String gatewayId) {
        if (this.queueDirectory == null) {
            return true;
        }
        if (gatewayId == null) {
            return this.emptyDirectory(this.pendingMessageDir, false);
        }
        return this.emptyDirectory(new File(this.pendingMessageDir, gatewayId), true);
    }

    private boolean storeDelayedMessage(OutboundMessage message) {
        if (this.queueDirectory == null) {
            return true;
        }
        return this.serializeMessage(message, new File(this.delayedMessageDir, message.getUuid() + MESSAGE_FILE_EXT));
    }

    private boolean deleteDelayedMessage(String messageUUID) {
        if (this.queueDirectory == null) {
            return true;
        }
        return new File(this.delayedMessageDir, messageUUID + MESSAGE_FILE_EXT).delete();
    }

    @Override
    public OutboundMessage pollDelayedMessage() {
        try {
            OutboundMessage message = ((ScheduledOutboundMessage)this.delayQueue.take()).getMessage();
            this.deleteDelayedMessage(message.getUuid());
            return message;
        }
        catch (InterruptedException interruptedException) {
            return null;
        }
    }

    @Override
    public OutboundMessage pollPendingMessage(String gatewayId) {
        PriorityBlockingQueue<OutboundMessage> queue = this.queueMap.get(gatewayId);
        if (queue == null) {
            return null;
        }
        OutboundMessage message = queue.poll();
        if (message != null) {
            this.deletePendingMessage(gatewayId, message.getUuid());
        }
        return message;
    }

    private boolean serializeMessage(OutboundMessage message, File toFile) {
        if (this.queueDirectory == null) {
            return true;
        }
        if (toFile.exists()) {
            Logger.getInstance().logError("Cannot save Message " + message.getUuid() + " File already exist.", null, null);
            return false;
        }
        ObjectOutputStream out = null;
        try {
            out = new ObjectOutputStream(new FileOutputStream(toFile));
            out.writeObject(message);
            out.close();
        }
        catch (IOException e) {
            Logger.getInstance().logError("Cannot save Message " + message.getUuid(), e, null);
            return false;
        }
        return true;
    }

    private OutboundMessage deserializeMessage(File fromFile) {
        if (!fromFile.exists()) {
            Logger.getInstance().logError("File of queued message doesn't exist " + fromFile.getPath(), null, null);
            return null;
        }
        ObjectInputStream in = null;
        OutboundMessage message = null;
        try {
            in = new ObjectInputStream(new FileInputStream(fromFile));
            message = (OutboundMessage)in.readObject();
            in.close();
        }
        catch (IOException e) {
            Logger.getInstance().logError("Could not read queued message from file " + fromFile.getPath(), e, null);
            return null;
        }
        catch (ClassNotFoundException e) {
            Logger.getInstance().logError("Could not read queued message from file " + fromFile.getPath(), e, null);
            return null;
        }
        return message;
    }

    private void loadPendingMessages() {
        File[] pendingDirs;
        for (File pendingDir : pendingDirs = this.pendingMessageDir.listFiles()) {
            File[] pendingFiles;
            if (!pendingDir.isDirectory()) continue;
            for (File pendingFile : pendingFiles = pendingDir.listFiles()) {
                if (!pendingFile.getName().endsWith(MESSAGE_FILE_EXT)) continue;
                this.addToGatewayQueue(this.deserializeMessage(pendingFile), false);
            }
        }
    }

    private void loadDelayedMessages() {
        File[] delayedFiles;
        for (File delayedFile : delayedFiles = this.delayedMessageDir.listFiles()) {
            if (!delayedFile.getName().endsWith(MESSAGE_FILE_EXT)) continue;
            OutboundMessage message = this.deserializeMessage(delayedFile);
            if (message.getDeliveryDelay() > 0L) {
                this.addToDelayedQueue(message, false);
                continue;
            }
            this.addToGatewayQueue(message, true);
            this.deleteDelayedMessage(message.getUuid());
        }
    }

    @Override
    public int delayedQueueSize(String gatewayId) {
        return this.delayQueue.size();
    }

    @Override
    public Collection<OutboundMessage> getDelayedMessages() {
        ArrayList<OutboundMessage> messages = new ArrayList<OutboundMessage>();
        for (ScheduledOutboundMessage scheduledOutboundMessage : this.delayQueue) {
            messages.add(scheduledOutboundMessage.getMessage());
        }
        return messages;
    }

    @Override
    public Collection<OutboundMessage> getPendingMessages(String gatewayId) {
        PriorityBlockingQueue<OutboundMessage> queue = this.queueMap.get(gatewayId);
        if (queue == null) {
            return new ArrayList<OutboundMessage>();
        }
        return new ArrayList<OutboundMessage>(queue);
    }

    @Override
    public int pendingQueueSize(String gatewayId) {
        PriorityBlockingQueue<OutboundMessage> queue = this.queueMap.get(gatewayId);
        if (queue == null) {
            return 0;
        }
        return queue.size();
    }

    @Override
    public boolean removeDelayedMessage(OutboundMessage message) {
        for (ScheduledOutboundMessage scheduledOutboundMessage : this.delayQueue) {
            if (!message.equals(scheduledOutboundMessage.getMessage()) || !this.delayQueue.remove(scheduledOutboundMessage)) continue;
            this.deleteDelayedMessage(message.getUuid());
            return true;
        }
        return false;
    }

    @Override
    public boolean removeDelayedMessage(String messageUUID) {
        for (ScheduledOutboundMessage scheduledOutboundMessage : this.delayQueue) {
            if (!messageUUID.equals(scheduledOutboundMessage.getMessage().getUuid()) || !this.delayQueue.remove(scheduledOutboundMessage)) continue;
            this.deleteDelayedMessage(messageUUID);
            return true;
        }
        return false;
    }

    @Override
    public boolean removeAllDelayedMessages() {
        this.delayQueue.clear();
        if (this.queueDirectory == null) {
            return true;
        }
        return this.emptyDirectory(this.delayedMessageDir, false);
    }

    @Override
    public boolean removeAllPendingMessages(String gatewayId) {
        PriorityBlockingQueue<OutboundMessage> queue = this.queueMap.get(gatewayId);
        if (queue != null) {
            queue.clear();
            this.queueMap.remove(queue);
            this.deletePendingMessages(gatewayId);
            return true;
        }
        return false;
    }

    @Override
    public boolean removeAllPendingMessages() {
        this.queueMap.clear();
        this.deletePendingMessages(null);
        return false;
    }

    private boolean emptyDirectory(File dir, boolean removeDir) {
        File[] pendingDirs;
        for (File file : pendingDirs = dir.listFiles()) {
            if (file.isDirectory()) {
                this.emptyDirectory(file, true);
                continue;
            }
            if (file.delete()) continue;
            return false;
        }
        if (removeDir) {
            return dir.delete();
        }
        return true;
    }

    class PriorityComparator
    implements Comparator<OutboundMessage> {
        PriorityComparator() {
        }

        @Override
        public int compare(OutboundMessage x, OutboundMessage y) {
            int comp = y.getPriority() - x.getPriority();
            if (comp == 0) {
                comp = x.getDate().compareTo(y.getDate());
            }
            return comp;
        }
    }
}

